/* main.c - FlexPWM example for S32R274 */
/* Description:  Generates PWM signal with FlexPWM */
/* Rev 1.0 Aug 16 2018 D Chung - production version */
/* Copyright NXP Semiconductor, Inc 2018 All rights reserved. */

/*******************************************************************************
* NXP Semiconductor Inc.
* (c) Copyright 2018 NXP Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without
any warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be
held liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED TO THE PROJECT
BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. IN NO EVENT
SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF
THIS AGREEMENT.

CUSTOMER agrees to hold NXP harmless against any and all claims demands or
actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result of the advise
or assistance supplied CUSTOMER in connection with product, services or goods
supplied under this Agreement.

Copyright 2018 NXP.  This software is owned or controlled by NXP and may only
be used strictly in accordance with the applicable license terms.  By expressly
accepting such terms or by downloading, installing, activating and/or otherwise
using the software, you are agreeing that you have read, and that you agree to
comply with and are bound by, such license terms.  If you do not agree to be
bound by the applicable license terms, then you may not retain, install, activate
or otherwise use the software.
********************************************************************************
* File              main_Z4.c
* Owner             David Chung
* Version           1.0
* Date              Aug-16-2018
* Classification    General Business Information
* Brief            	FlexPWM generation
*                    RUN0 mode with max core frequency(200MHz) generated from PLL1
********************************************************************************
* Detailed Description:
* This example shows usage of FlexPWM module. The FlexPWM is very useful for
* generating the phase outputs of a motor.
*
* This example is set for 2.44140625 kHz PWM frequency.
*
* NOTE: There is an issue with the P&E Micro USB Multilink debugger where if you
* are debugging this project with that debugger, the incorrect FlexPWM registers
* would be written to.  In other words, an instruction for writing to Reg(A) will
* instead write to Reg(B).  However, if you program the board using P&E Micro USB
* Multilink and then disconnect from the debugger by resetting the board, the demo
* will run as intended.
*
* Lauterbach has not been observed to exhibit the same issue as P&E Micro USB
* Multilink.  Debugging with Lauterbach will yield the intended FlexPWM signal.
* ------------------------------------------------------------------------------
* Test HW:         S32R274RRUEVB+MPC57xxMB
* MCU:             S32R274
* Terminal:        None
* Fsys:            200 MHz PLL on 40MHz external oscillator
* Debugger:        JTAG (e.g. Lauterbach, USB Multilink)
* Target:          FLASH
* EVB connection:  A[11] (P8.12, MPC57xxMB).. FlexPWM A[0] output to oscilloscope
* 				   A[10] (P8.11, MPC57xxMB).. FlexPWM B[0] output to oscilloscope
*
********************************************************************************
Revision History:
Version  Date         Author  			Description of Changes
1.0      Aug-16-2018  David Chung	  	Initial version

*******************************************************************************/

/*******************************************************************************
* Includes
*******************************************************************************/
#include "derivative.h" /* include peripheral declarations */

/*******************************************************************************
* Constants and macros
*******************************************************************************/

#define AUX0_clk_DIV1 	7
#define SGEN_IOFREQ 	128

#define AUX0_clk_DIV0 	0
#define PWM_PRESCALER	4
#define PWM_MODULO 	4096

#define KEY_VALUE1 0x5AF0ul
#define KEY_VALUE2 0xA50Ful

/*******************************************************************************
* External objects
*******************************************************************************/
extern void xcptn_xmpl(void);

/*******************************************************************************
* Global variables
*******************************************************************************/
//extern const uint32_t IntcIsrVectorTable[];
//extern void xcptn_xmpl(void);

/*******************************************************************************
* Local functions
*******************************************************************************/
void hw_init(void)
{
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	uint32_t mctl = MC_ME.MCTL.R;
#endif
#if defined(TURN_ON_CPU1)
	/* enable core 1 in all modes */
	MC_ME.CCTL2.R = 0x00FE;
	/* Set Start address for core 1: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR2.R = 0x1080000 | 0x1;
#else
    MC_ME.CADDR2.R = 0x4006a800 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif	
#if defined(TURN_ON_CPU2)
	/* enable core 2 in all modes */
	MC_ME.CCTL3.R = 0x00FE;
	/* Set Start address for core 2: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR3.R = 0x1100000 | 0x1;
#else
    MC_ME.CADDR3.R = 0x400d5000 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	MC_ME.MCTL.R = (mctl & 0xffff0000ul) | KEY_VALUE1;
	MC_ME.MCTL.R =  mctl; /* key value 2 always from MCTL */
#endif
}

/*******************************************************************************
Function Name : SysClk_Init
Engineer      : David Chung
Date          : Aug-16-2018
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : Enable XOSC, PLL0, PLL1 and enter RUN0 with PLL1 as sys clk (200 MHz)
Issues        : NONE
*******************************************************************************/
void SysClk_Init(void)
{
    CGM.AC3_SC.B.SELCTL = 0x01;		    //connect XOSC to the PLL0 input
    CGM.AC4_SC.B.SELCTL = 0x01;		    //connect XOSC to the PLL1 input

    // Set PLL0 to 160 MHz with 40MHz XOSC reference
    PLLDIG.PLL0DV.R = 0x28021008;	     // PREDIV =  1, MFD = 8, RFDPHI = 2, RFDPHI1 = 5
    MC_ME.RUN_MC[0].R = 0x00130070;		    // RUN0 cfg: IRCON,OSC0ON,PLL0ON,syclk=IRC

    // Mode Transition to enter RUN0 mode:
    MC_ME.MCTL.R = 0x40005AF0;		    // Enter RUN0 Mode & Key
    MC_ME.MCTL.R = 0x4000A50F;		    // Enter RUN0 Mode & Inverted Key
    while (MC_ME.GS.B.S_MTRANS) {};		    // Wait for mode transition to complete
    while(MC_ME.GS.B.S_CURRENT_MODE != 4) {};	    // Verify RUN0 is the current mode

    // Set PLL1 to 200 MHz with 40MHz XOSC reference
    PLLDIG.PLL1DV.R = 0x00020014;	     // MFD = 20, RFDPHI = 2
//    MC_ME.RUN_PC[0].R = 0x000000FE;		    // enable peripherals run in all modes
    MC_ME.RUN_PC[1].R = 0x000000FE;			//Enable peripherals to run in all modes if they follow RUN_PC[1]
    MC_ME.PCTL[255].B.RUN_CFG = 1;			//Make FlexPWM0 follow RUN_PC[1]
    MC_ME.RUN_MC[0].R = 0x001300F4;		    // RUN0 cfg: IRCON, OSC0ON, PLL1ON, syclk=PLL1
    CGM.SC_DC0.R = 0x80030000;    // SYS_CLK at syst clk div by 4 ... (50 MHz)

    // Mode Transition to enter RUN0 mode:
    MC_ME.MCTL.R = 0x40005AF0;		    // Enter RUN0 Mode & Key
    MC_ME.MCTL.R = 0x4000A50F;		    // Enter RUN0 Mode & Inverted Key
    while (MC_ME.GS.B.S_MTRANS) {};		    // Wait for mode transition to complete
    while(MC_ME.GS.B.S_CURRENT_MODE != 4) {};	    // Verify RUN0 is the current mode

    // set peripheral clocks
    CGM.AC0_SC.R = 0x02000000;    // Select PLL0_PHI_CLK for auxiliary clock 0
    CGM.AC0_DC0.R = 0x80000000|(AUX0_clk_DIV0<<16);    // MOTC_CLK : Enable aux clk 0 div by (AUX0_clk_DIV0+1) (160 MHz)
}

/*******************************************************************************
Function Name : SIUL2_Init
Engineer      : David Chung
Date          : Aug-16-2018
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : SIUL2 initialization (ports)
Issues        : NONE
*******************************************************************************/
void SIUL2_Init(void)
{
    SIUL2.MSCR[11].R = 0x02800002; // A[11] pin as FlexPWM A[0] output
    SIUL2.MSCR[10].R = 0x02800002; // A[10] pin as FlexPWM B[0] output

}

/*******************************************************************************
Function Name : FlexPWM_Init
Engineer      : PetrS
Date          : Apr-14-2015
Parameters    : NONE
Modifies      : NONE
Returns       : NONE
Notes         : FlexPWM initialization
Issues        : NONE
*******************************************************************************/
static void FlexPWM_Init(void)
{
    /* Submodule 0 Initialisation */
    FlexPWM_0.OUTEN.R           = 0x110;	// enable A and B outputs on submodule 0

    FlexPWM_0.SUB[0].CTRL1.R    = 0x0400|(PWM_PRESCALER<<4);	// full cycle reload, every opportunity
    FlexPWM_0.SUB[0].CTRL2.R    = 0x2000;	// independent outputs
    FlexPWM_0.SUB[0].DTCNT0.R   = 0x0000;	// deadtime values
    FlexPWM_0.SUB[0].DTCNT1.R   = 0x0000;
    FlexPWM_0.SUB[0].INIT.R   = 0x1;
    FlexPWM_0.SUB[0].VAL1.R   = PWM_MODULO;	// PWM modulo
    FlexPWM_0.SUB[0].VAL2.R   = 0x1;   		// PWM A0 rising edge
    FlexPWM_0.SUB[0].VAL3.R   = PWM_MODULO/3;	// PWM A0 falling edge
    FlexPWM_0.SUB[0].VAL4.R   = PWM_MODULO/2;	// PWM B0 rising edge
//	FlexPWM_0.SUB[0].VAL4.R = 0x1;
    FlexPWM_0.SUB[0].VAL5.R   = PWM_MODULO;	// PWM B0 falling edge
    FlexPWM_0.SUB[0].DISMAP.R   = 0x0000;	// disable fault pin condition

    FlexPWM_0.MCTRL.B.LDOK = 0x1;	/* Load the PRSC bits of CTRL1 and the INIT, and VALx registers
    								   into a set of buffers */

}

int main(void)
{
    unsigned short i=1,j = 0;

    xcptn_xmpl();

    SysClk_Init();
    SIUL2_Init();
    FlexPWM_Init();

    FlexPWM_0.MCTRL.B.RUN = 0x1;		// Submodule 0 PWM generator enabled


    while(1)
    {
#if 1
	for(j=0;j<10000;j++){}

	if(i<PWM_MODULO){
		FlexPWM_0.SUB[0].VAL3.R = i;
//		FlexPWM_0.SUB[0].VAL5.R = i;
	}else{
	    if(i<2*PWM_MODULO){
	    	FlexPWM_0.SUB[0].VAL3.R = (2*PWM_MODULO) - i;
//			FlexPWM_0.SUB[0].VAL5.R = (2*PWM_MODULO) - i;
	    }else{
	    	i=1;
	    }
	}
	i++;
	FlexPWM_0.MCTRL.B.LDOK = 0x1;
#endif

    }
    return 0;
}


